# 機能設計書 90-Feature Column API

## 概要

本ドキュメントは、TensorFlowのFeature Column API（特徴量変換API）について、その設計・処理内容・インタフェース仕様を記述した機能設計書である。数値・カテゴリカル・バケット化・クロス特徴量などの特徴量変換を提供する。

### 本機能の処理概要

Feature Column APIは、入力データの各特徴量を機械学習モデルに適した形式に変換する高水準APIである。数値データ、カテゴリカルデータ、テキストデータなど多様なデータ型を統一的なインタフェースで処理し、モデルへの入力層を構築する。

**業務上の目的・背景**：機械学習モデルの入力データは多様な形式（数値、カテゴリカル、テキスト、バケット化等）を持つ。これらを適切なテンソル表現に変換する処理は頻繁に必要とされるが、手動実装は煩雑でエラーを起こしやすい。Feature Column APIは、宣言的な特徴量定義により変換処理を自動化する。ただし、本APIは非推奨であり、Keras前処理層（tf.keras.layers）またはtf.keras.utils.FeatureSpaceへの移行が推奨されている。

**機能の利用シーン**：構造化データ（テーブルデータ）の前処理、tf.estimator APIとの組み合わせ、DNN+Wideモデルの特徴量入力、既存コードのメンテナンスなど。

**主要な処理内容**：
1. numeric_column: 数値特徴量の定義
2. bucketized_column: バケット化（数値→カテゴリカル変換）
3. categorical_column_with_vocabulary_list/file: 語彙ベースカテゴリカル
4. categorical_column_with_hash_bucket: ハッシュバケットカテゴリカル
5. categorical_column_with_identity: ID直接マッピング
6. crossed_column: 特徴量クロス（交互作用）
7. embedding_column: カテゴリカル→密ベクトル埋め込み
8. indicator_column: ワンホットエンコーディング
9. shared_embedding_columns: 共有埋め込みテーブル
10. StateManagerによる変数・リソース管理

**関連システム・外部連携**：Kerasモデル構築API（No.131）、埋め込み層（No.20）、tf.estimator APIと連携する。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | バックエンドの特徴量変換機能であり、直接的な画面関連はない |

## 機能種別

データ変換（特徴量エンジニアリング）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| key | string | Yes | 特徴量のキー名 | 非空文字列 |
| shape | tuple | No | 数値特徴量の形状 | デフォルト: (1,) |
| default_value | float/int | No | 欠損値のデフォルト値 | dtype互換 |
| dtype | DType | No | データ型 | float32/int64等 |
| vocabulary_list | list[str] | No | 語彙リスト | カテゴリカル変換時 |
| hash_bucket_size | int | No | ハッシュバケット数 | > 0 |
| boundaries | list[float] | No | バケット境界値リスト | 昇順 |
| dimension | int | No | 埋め込み次元数 | > 0 |
| combiner | string | No | 埋め込み結合方法 | 'mean'/'sqrtn'/'sum' |

### 入力データソース

構造化データ（辞書形式のテンソル/SparseTensor/RaggedTensor）。tf.data Dataset APIからのバッチデータ。CSVファイル、TFRecord等。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| dense_tensor | Tensor | 変換後の密テンソル（input_layer経由） |
| feature_column | FeatureColumn | 特徴量カラムオブジェクト |
| state | Variable/Resource | StateManagerで管理される変数・リソース |

### 出力先

Kerasモデルの入力層、tf.estimatorの特徴量入力。

## 処理フロー

### 処理シーケンス

```
1. 特徴量カラム定義
   └─ numeric_column / categorical_column_with_* 等で宣言的に定義
2. StateManager初期化
   └─ 変数・リソースの管理オブジェクト生成
3. 特徴量変換実行
   └─ input_layer(features, feature_columns) で一括変換
4. 数値特徴量処理
   └─ 正規化、バケット化
5. カテゴリカル特徴量処理
   └─ 語彙検索、ハッシュ化、ワンホット/埋め込み
6. 結合
   └─ 全特徴量を結合して密テンソルを生成
```

### フローチャート

```mermaid
flowchart TD
    A[特徴量カラム定義] --> B[入力データ受け取り]
    B --> C{特徴量タイプ判定}
    C -->|数値| D[numeric_column処理]
    C -->|カテゴリカル| E[categorical_column処理]
    C -->|バケット化| F[bucketized_column処理]
    C -->|クロス| G[crossed_column処理]
    D --> H{埋め込み/指標化?}
    E --> H
    F --> H
    G --> H
    H -->|埋め込み| I[embedding_column処理]
    H -->|指標| J[indicator_column処理]
    H -->|直接| K[密テンソル出力]
    I --> K
    J --> K
    K --> L[全特徴量結合]
    L --> M[モデル入力テンソル]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-90-01 | 非推奨警告 | Feature Column APIは非推奨。Keras前処理層への移行を推奨 | 常時 |
| BR-90-02 | FeatureColumnステートレス | FeatureColumnは設定のみを保持し、状態はStateManagerが管理 | 常時 |
| BR-90-03 | 語彙検索 | vocabulary_list/file指定時はハッシュテーブルによる検索 | カテゴリカル変換時 |
| BR-90-04 | 埋め込み次元 | 埋め込み次元のデフォルトは`dimension = int(num_buckets ** 0.25)` | embedding_column使用時 |

### 計算ロジック

- **ハッシュバケット**: `bucket_id = hash(value) % hash_bucket_size`
- **バケット化**: `bucket_id = bisect(boundaries, value)`
- **クロス特徴量**: `cross_id = hash(feature_a + "_X_" + feature_b) % hash_bucket_size`
- **埋め込み結合**: mean→平均、sqrtn→sqrt(N)で除算、sum→合計

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（インメモリ処理。語彙ファイルのファイルシステム読み込みあり）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | 入力エラー | 不正なカラム定義（空key、負のdimension等） | パラメータを修正 |
| KeyError | 入力エラー | features辞書にkeyが存在しない | 入力データのキーを確認 |
| TypeError | 型エラー | 不正なデータ型の入力 | dtypeパラメータを確認 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 語彙検索はハッシュテーブルで定数時間
- 大規模語彙ファイルの読み込みは初期化時のコスト

## セキュリティ考慮事項

- 語彙ファイルのパスは信頼できるソースから指定すること
- ハッシュベースの特徴量はハッシュ衝突による精度低下に注意

## 備考

- 本APIは非推奨（deprecated）であり、tf.keras.layers（Keras前処理層）またはtf.keras.utils.FeatureSpaceへの移行が推奨される
- tf.feature_column.* の各関数にはtf.keras.layersの対応する前処理層が存在する

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | feature_column_v2.py | `tensorflow/python/feature_column/feature_column_v2.py` | StateManagerクラス、FeatureColumn基底クラス |

**読解のコツ**: Feature Column APIは「宣言的プログラミング」パターンを採用している。各カラム定義はデータ変換の「仕様」を宣言し、実際の変換はinput_layer呼び出し時に実行される。StateManagerが変数のライフサイクル管理を担当する。

#### Step 2: StateManagerを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | feature_column_v2.py | `tensorflow/python/feature_column/feature_column_v2.py` | StateManagerクラス定義 |

**主要処理フロー**:
1. **92-98行目**: StateManagerクラスの概要（FeatureColumnはステートレス、StateManagerが状態管理）
2. **101-120行目**: create_variable - 特徴量カラムに対応する変数生成

#### Step 3: 非推奨警告を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | feature_column_v2.py | `tensorflow/python/feature_column/feature_column_v2.py` | 非推奨定義とマイグレーションガイド |

**主要処理フロー**:
- **73-76行目**: _FEATURE_COLUMN_DEPRECATION定数 - 非推奨メッセージ
- **77-84行目**: _FEATURE_COLUMN_DEPRECATION_WARNING - 詳細な移行ガイドURL
- **85-89行目**: _FEATURE_COLUMN_DEPRECATION_RUNTIME_WARNING - ランタイム警告メッセージ

#### Step 4: 各カラム実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | feature_column_v2.py | `tensorflow/python/feature_column/feature_column_v2.py` | numeric_column、categorical_column_with_*、embedding_column等の実装 |

### プログラム呼び出し階層図

```
tf.feature_column.numeric_column('price')
    |
    +-- NumericColumn オブジェクト生成 (宣言的定義)

tf.feature_column.embedding_column(categorical_col, dimension=8)
    |
    +-- EmbeddingColumn オブジェクト生成

tf.keras.layers.DenseFeatures(feature_columns)
    |
    +-- StateManager初期化
    |       +-- create_variable (各カラムの変数生成)
    |
    +-- __call__(features)
            +-- NumericColumn.get_dense_tensor()
            |       +-- 数値変換・正規化
            |
            +-- CategoricalColumn.get_sparse_tensors()
            |       +-- 語彙検索 / ハッシュ化
            |
            +-- EmbeddingColumn.get_dense_tensor()
            |       +-- embedding_ops.safe_embedding_lookup_sparse
            |
            +-- 全特徴量の結合
```

### データフロー図

```
[入力]                        [処理]                            [出力]

特徴量辞書 ────────▶ FeatureColumn定義に基づく変換
{                           |
  'price': [10.0],    数値 → NumericColumn ──────▶ 密テンソル
  'color': ['red'],   カテゴリ → CategoricalColumn     |
  'age': [25],        バケット → BucketizedColumn      |
}                           |                           v
                     EmbeddingColumn ──────────▶ 埋め込みベクトル
                            |                           |
                            v                           v
                     全特徴量結合 ────────────▶ モデル入力テンソル
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| feature_column_v2.py | `tensorflow/python/feature_column/feature_column_v2.py` | ソース | Feature Column API本体 |
| feature_column_v2_types.py | `tensorflow/python/feature_column/feature_column_v2_types.py` | ソース | 型定義 |
| feature_column.py | `tensorflow/python/feature_column/feature_column.py` | ソース | V1 API（レガシー） |
| serialization.py | `tensorflow/python/feature_column/serialization.py` | ソース | シリアライゼーション |
| utils.py | `tensorflow/python/feature_column/utils.py` | ソース | ユーティリティ |
